// AI Content Rewriter - Content Script
class AIContentRewriter {
  constructor() {
    this.buttons = new Map();
    this.activeModal = null;
    this.pluginEnabled = true; // Default to enabled
    this.init();
  }

  init() {
    this.checkPluginState().then(() => {
      // Only inject buttons and setup observers if plugin is enabled
      if (this.pluginEnabled) {
        this.injectButtons();
        this.setupObserver();
        this.setupKeyboardShortcuts();
      } else {
        console.log('Plugin disabled - skipping button injection and observers');
      }
    });
    
    // Listen for plugin toggle messages
    this.setupMessageListener();
  }

  setupMessageListener() {
    chrome.runtime.onMessage.addListener((request, sender, sendResponse) => {
      console.log('Content script received message:', request);
      
      if (request.action === 'togglePlugin') {
        console.log('Toggling plugin:', request.enabled);
        this.pluginEnabled = request.enabled;
        
        if (this.pluginEnabled) {
          console.log('Plugin enabled - injecting buttons');
          this.injectButtons();
        } else {
          console.log('Plugin disabled - removing buttons');
          this.removeAllButtons();
        }
        
        sendResponse({ success: true });
      }
    });
  }

  // Inject rewrite buttons next to input fields
  injectButtons() {
    if (!this.pluginEnabled) {
      console.log('Plugin disabled, not injecting buttons');
      return;
    }

    const inputs = document.querySelectorAll('input[type="text"], input[type="email"], textarea, [contenteditable="true"]');
    
    inputs.forEach(input => {
      if (this.shouldSkipInput(input)) return;
      
      const button = this.createRewriteButton(input);
      this.positionButton(button, input);
      this.buttons.set(input, button);
    });
  }

  shouldSkipInput(input) {
    // Skip password fields, hidden inputs, and already processed inputs
    if (input.type === 'password' || 
        input.type === 'hidden' || 
        input.style.display === 'none' ||
        this.buttons.has(input)) {
      return true;
    }
    
    // Skip very small inputs (likely not for content)
    const rect = input.getBoundingClientRect();
    return rect.width < 100 || rect.height < 20;
  }

  createRewriteButton(input) {
    const button = document.createElement('button');
    button.className = 'ai-rewriter-btn';
    button.innerHTML = `
      <svg width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
        <path d="M11 4H4a2 2 0 0 0-2 2v14a2 2 0 0 0 2 2h14a2 2 0 0 0 2-2v-7"/>
        <path d="m18.5 2.5 3 3L12 15l-4 1 1-4 9.5-9.5z"/>
        <path d="M16 5h3v3"/>
      </svg>
    `;
    button.title = 'Rewrite with AI (Ctrl+Shift+R)';
    
    button.addEventListener('click', (e) => {
      e.preventDefault();
      e.stopPropagation();
      this.handleRewriteClick(input);
    });

    return button;
  }

  positionButton(button, input) {
    document.body.appendChild(button);
    
    const rect = input.getBoundingClientRect();
    const scrollX = window.pageXOffset;
    const scrollY = window.pageYOffset;
    
    // Position button at the right edge of the input
    button.style.position = 'absolute';
    button.style.left = `${rect.right - 30 + scrollX}px`;
    button.style.top = `${rect.top + 5 + scrollY}px`;
    button.style.zIndex = '10000';
    
    // Update position on scroll/resize
    const updatePosition = () => {
      if (!document.contains(input)) {
        button.remove();
        this.buttons.delete(input);
        return;
      }
      
      const newRect = input.getBoundingClientRect();
      button.style.left = `${newRect.right - 30 + window.pageXOffset}px`;
      button.style.top = `${newRect.top + 5 + window.pageYOffset}px`;
    };
    
    window.addEventListener('scroll', updatePosition);
    window.addEventListener('resize', updatePosition);
  }

  setupObserver() {
    // Watch for dynamically added input fields
    const observer = new MutationObserver((mutations) => {
      // Skip if plugin is disabled
      if (!this.pluginEnabled) return;
      
      mutations.forEach((mutation) => {
        mutation.addedNodes.forEach((node) => {
          if (node.nodeType === Node.ELEMENT_NODE) {
            const inputs = node.querySelectorAll?.('input[type="text"], input[type="email"], textarea, [contenteditable="true"]') || 
                          (node.matches?.('input[type="text"], input[type="email"], textarea, [contenteditable="true"]') ? [node] : []);
            
            inputs.forEach(input => {
              if (!this.shouldSkipInput(input)) {
                const button = this.createRewriteButton(input);
                this.positionButton(button, input);
                this.buttons.set(input, button);
              }
            });
          }
        });
      });
    });

    observer.observe(document.body, { childList: true, subtree: true });
  }

  setupKeyboardShortcuts() {
    document.addEventListener('keydown', (e) => {
      // Skip if plugin is disabled
      if (!this.pluginEnabled) return;
      
      if (e.ctrlKey && e.shiftKey && e.key === 'R') {
        e.preventDefault();
        const activeElement = document.activeElement;
        if (this.buttons.has(activeElement)) {
          this.handleRewriteClick(activeElement);
        }
      }
    });
  }

  async handleRewriteClick(input) {
    const text = this.getInputText(input);
    if (!text.trim()) {
      this.showToast('Please enter some text to rewrite');
      return;
    }

    this.showRewriteModal(input, text);
  }

  getInputText(input) {
    if (input.contentEditable === 'true') {
      return input.textContent || '';
    }
    return input.value || '';
  }

  setInputText(input, text) {
    if (input.contentEditable === 'true') {
      input.textContent = text;
      // Trigger input event for frameworks
      input.dispatchEvent(new Event('input', { bubbles: true }));
    } else {
      input.value = text;
      input.dispatchEvent(new Event('input', { bubbles: true }));
      input.dispatchEvent(new Event('change', { bubbles: true }));
    }
  }

  showRewriteModal(input, originalText) {
    // Remove existing modal if any
    if (this.activeModal) {
      this.activeModal.remove();
    }

    const modal = document.createElement('div');
    modal.className = 'ai-rewriter-modal';
    modal.innerHTML = `
      <div class="ai-rewriter-modal-content">
        <div class="ai-rewriter-header">
          <h3>AI Content Rewriter</h3>
          <button class="ai-rewriter-close">&times;</button>
        </div>
        <div class="ai-rewriter-body">
          <div class="ai-rewriter-original">
            <label>Original Text:</label>
            <div class="ai-rewriter-text">${this.escapeHtml(originalText)}</div>
          </div>
          <div class="ai-rewriter-options">
            <label for="ai-rewriter-tone">Tone:</label>
            <select id="ai-rewriter-tone">
              <option value="professional">Professional</option>
              <option value="casual">Casual</option>
              <option value="creative">Creative</option>
              <option value="formal">Formal</option>
              <option value="friendly">Friendly</option>
            </select>
            <label for="ai-rewriter-length">Length:</label>
            <select id="ai-rewriter-length">
              <option value="shorter" selected>Shorter</option>
              <option value="same">Same Length</option>
              <option value="longer">Longer</option>
            </select>
          </div>
          <div class="ai-rewriter-actions">
            <button id="ai-rewriter-generate" class="ai-rewriter-btn-primary">
              Generate Rewrites
            </button>
          </div>
          <div class="ai-rewriter-results" id="ai-rewriter-results" style="display: none;">
            <label>Choose a rewrite:</label>
            <div class="ai-rewriter-samples"></div>
            <div class="ai-rewriter-result-actions">
              <button id="ai-rewriter-apply" class="ai-rewriter-btn-success">Apply Selected</button>
              <button id="ai-rewriter-regenerate" class="ai-rewriter-btn-secondary">Generate More</button>
            </div>
          </div>
        </div>
      </div>
    `;

    document.body.appendChild(modal);
    this.activeModal = modal;

    // Event listeners
    modal.querySelector('.ai-rewriter-close').addEventListener('click', () => {
      modal.remove();
      this.activeModal = null;
    });

    modal.addEventListener('click', (e) => {
      if (e.target === modal) {
        modal.remove();
        this.activeModal = null;
      }
    });

    modal.querySelector('#ai-rewriter-generate').addEventListener('click', () => {
      this.generateRewrites(originalText, input);
    });

    modal.querySelector('#ai-rewriter-apply').addEventListener('click', () => {
      this.applySelectedRewrite(input);
    });

    modal.querySelector('#ai-rewriter-regenerate').addEventListener('click', () => {
      this.generateRewrites(originalText, input);
    });
  }

  async generateRewrites(text, input) {
    const generateBtn = document.querySelector('#ai-rewriter-generate');
    const resultsDiv = document.querySelector('#ai-rewriter-results');
    const samplesDiv = document.querySelector('.ai-rewriter-samples');
    
    // Show loading state
    generateBtn.textContent = 'Generating...';
    generateBtn.disabled = true;
    
    const tone = document.querySelector('#ai-rewriter-tone').value;
    const length = document.querySelector('#ai-rewriter-length').value;

    try {
      // Get current tab ID for rate limiting (with fallback)
      let tabId = 'unknown';
      try {
        if (chrome.tabs && chrome.tabs.query) {
          const [tab] = await chrome.tabs.query({ active: true, currentWindow: true });
          tabId = tab?.id || 'unknown';
        }
      } catch (tabError) {
        console.warn('Could not get tab ID:', tabError);
        tabId = 'unknown';
      }
      
      // Send message to background script for API call
      const response = await chrome.runtime.sendMessage({
        action: 'rewriteText',
        text: text,
        tone: tone,
        length: length,
        tabId: tabId
      });

      if (response.success) {
        this.displayRewrites(response.rewrites, samplesDiv);
        resultsDiv.style.display = 'block';
      } else {
        this.showToast(`Error: ${response.error}`, 'error');
      }
    } catch (error) {
      console.error('Rewrite error details:', {
        error: error.message,
        stack: error.stack,
        text: text,
        tone: tone,
        length: length
      });
      
      if (error.message.includes('No API key found')) {
        this.showToast('Please save your API key in the extension popup first', 'error');
      } else if (error.message.includes('Rate limit')) {
        this.showToast('Rate limit exceeded. Please wait before trying again.', 'error');
      } else {
        this.showToast(`Error: ${error.message}`, 'error');
      }
    } finally {
      generateBtn.textContent = 'Generate Rewrites';
      generateBtn.disabled = false;
    }
  }

  displayRewrites(rewrites, container) {
    container.innerHTML = '';
    
    rewrites.forEach((rewrite, index) => {
      const option = document.createElement('div');
      option.className = 'ai-rewriter-option';
      option.innerHTML = `
        <input type="radio" name="rewrite-choice" value="${index}" id="rewrite-${index}">
        <label for="rewrite-${index}" class="ai-rewriter-option-text">
          ${this.escapeHtml(rewrite.trim())}
        </label>
      `;
      container.appendChild(option);
    });

    // Select first option by default
    if (rewrites.length > 0) {
      container.querySelector('input[type="radio"]').checked = true;
    }
  }

  applySelectedRewrite(input) {
    const selectedRadio = document.querySelector('input[name="rewrite-choice"]:checked');
    if (!selectedRadio) {
      this.showToast('Please select a rewrite option');
      return;
    }

    const rewriteIndex = parseInt(selectedRadio.value);
    const samplesDiv = document.querySelector('.ai-rewriter-samples');
    const rewriteText = samplesDiv.children[rewriteIndex].querySelector('.ai-rewriter-option-text').textContent.trim();
    
    this.setInputText(input, rewriteText);
    this.activeModal.remove();
    this.activeModal = null;
    
    this.showToast('Text rewritten successfully!', 'success');
  }

  showToast(message, type = 'info') {
    const toast = document.createElement('div');
    toast.className = `ai-rewriter-toast ai-rewriter-toast-${type}`;
    toast.textContent = message;
    
    document.body.appendChild(toast);
    
    // Animate in
    setTimeout(() => toast.classList.add('ai-rewriter-toast-show'), 100);
    
    // Remove after 3 seconds
    setTimeout(() => {
      toast.classList.remove('ai-rewriter-toast-show');
      setTimeout(() => toast.remove(), 300);
    }, 3000);
  }

  escapeHtml(text) {
    const div = document.createElement('div');
    div.textContent = text;
    return div.innerHTML;
  }

  async checkPluginState() {
    try {
      const result = await chrome.storage.local.get('userSettings');
      const settings = result.userSettings || {};
      this.pluginEnabled = settings.pluginEnabled !== false; // Default to true
      console.log('Plugin state loaded:', this.pluginEnabled);
    } catch (error) {
      console.error('Error checking plugin state:', error);
      this.pluginEnabled = true; // Default to enabled on error
    }
  }

  removeAllButtons() {
    console.log('Removing all buttons, count:', this.buttons.size);
    this.buttons.forEach(button => {
      button.remove();
    });
    this.buttons.clear();
  }
}

// Initialize the content rewriter when DOM is ready
if (document.readyState === 'loading') {
  document.addEventListener('DOMContentLoaded', () => new AIContentRewriter());
} else {
  new AIContentRewriter();
}