// AI Content Rewriter - Background Service Worker
class AIRewriterBackground {
  constructor() {
    this.setupMessageListener();
    this.setupTabListener();
    this.setupInstallListener();
    this.rateLimiter = new Map();
    this.apiProviders = {
      openai: new OpenAIProvider()
    };
    
    // Setup periodic cleanup
    setInterval(() => {
      this.cleanupRateLimiter();
    }, 60000); // Clean up every minute
    
    // Sync plugin state with all existing tabs on startup
    this.syncPluginStateWithAllTabs();
    
    // Also sync when the service worker starts (for extension restarts)
    chrome.runtime.onStartup.addListener(() => {
      this.syncPluginStateWithAllTabs();
    });
    
    // Also sync when the service worker is installed (for extension restarts)
    self.addEventListener('install', () => {
      this.syncPluginStateWithAllTabs();
    });
    
    // Also sync when the service worker is activated (for extension restarts)
    self.addEventListener('activate', () => {
      this.syncPluginStateWithAllTabs();
    });
  }

  setupMessageListener() {
    chrome.runtime.onMessage.addListener((request, sender, sendResponse) => {
      if (request.action === 'rewriteText') {
        this.handleRewriteRequest(request, sendResponse);
        return true; // Keep message channel open for async response
      } else if (request.action === 'saveSettings') {
        this.handleSaveSettings(request, sendResponse);
        return true; // Keep message channel open for async response
      } else if (request.action === 'getSettings') {
        this.handleGetSettings(request, sendResponse);
        return true; // Keep message channel open for async response
      } else if (request.action === 'saveRules') {
        this.handleSaveRules(request, sendResponse);
        return true; // Keep message channel open for async response
      } else if (request.action === 'getRules') {
        this.handleGetRules(request, sendResponse);
        return true; // Keep message channel open for async response
      } else if (request.action === 'togglePlugin') {
        this.handleTogglePlugin(request, sendResponse);
        return true; // Keep message channel open for async response
      }
    });
  }

  setupTabListener() {
    // Listen for tab updates to sync plugin state with new tabs
    chrome.tabs.onUpdated.addListener((tabId, changeInfo, tab) => {
      if (changeInfo.status === 'complete' && tab.url && tab.url.startsWith('http')) {
        // Tab has finished loading, sync plugin state
        this.syncPluginStateWithTab(tabId);
      }
    });
  }

  setupInstallListener() {
    // Listen for extension installation/update
    chrome.runtime.onInstalled.addListener(async (details) => {
      if (details.reason === 'install' || details.reason === 'update') {
        // Wait a bit for the extension to fully initialize
        setTimeout(async () => {
          // Sync plugin state with all existing tabs
          const tabs = await chrome.tabs.query({});
          tabs.forEach(tab => {
            if (tab.url && tab.url.startsWith('http')) {
              this.syncPluginStateWithTab(tab.id);
            }
          });
        }, 2000);
      }
    });
  }

  async handleRewriteRequest(request, sendResponse) {
    try {
      console.log('Rewrite request received:', request);
      
      // Check rate limiting
      const tabId = request.tabId || 'unknown';
      if (!this.checkRateLimit(tabId)) {
        sendResponse({
          success: false,
          error: 'Rate limit exceeded. Please wait before making another request.'
        });
        return;
      }

      // Get user settings
      const settings = await this.getStoredSettings();
      console.log('Settings loaded:', { 
        provider: settings.provider, 
        hasApiKey: !!settings.apiKey,
        apiKeyLength: settings.apiKey ? settings.apiKey.length : 0,
        apiKeyPrefix: settings.apiKey ? settings.apiKey.substring(0, 15) + '...' : 'none'
      });
      
      // Get custom rules
      const rules = await this.getStoredRules();
      console.log('Rules loaded:', rules);
      
      if (!settings.apiKey) {
        sendResponse({
          success: false,
          error: 'No API key found. Please save your settings first.'
        });
        return;
      }
      
      const provider = this.apiProviders[settings.provider];
      
      if (!provider) {
        sendResponse({
          success: false,
          error: `Invalid AI provider selected: ${settings.provider}`
        });
        return;
      }

      console.log('Generating rewrites with provider:', settings.provider);
      console.log('Provider class:', provider.constructor.name);
      
      // Generate rewrites with custom rules
      const rewrites = await provider.generateRewrites(
        request.text,
        request.tone,
        request.length,
        settings.apiKey,
        rules,
        settings.maxSamples
      );

      console.log('Rewrites generated successfully:', rewrites.length);
      sendResponse({
        success: true,
        rewrites: rewrites
      });

    } catch (error) {
      console.error('Rewrite request error:', error);
      console.error('Error stack:', error.stack);
      sendResponse({
        success: false,
        error: error.message || 'Failed to generate rewrites'
      });
    }
  }

  async handleSaveRules(request, sendResponse) {
    try {
      await chrome.storage.local.set({ userRules: request.rules });
      sendResponse({ success: true });
    } catch (error) {
      sendResponse({ success: false, error: error.message });
    }
  }

  async handleGetRules(request, sendResponse) {
    try {
      const rules = await this.getStoredRules();
      sendResponse({ success: true, rules });
    } catch (error) {
      sendResponse({ success: false, error: error.message });
    }
  }

  async handleSaveSettings(request, sendResponse) {
    try {
      await chrome.storage.local.set({ userSettings: request.settings });
      sendResponse({ success: true });
    } catch (error) {
      sendResponse({ success: false, error: error.message });
    }
  }

  async handleGetSettings(request, sendResponse) {
    try {
      const settings = await this.getStoredSettings();
      sendResponse({ success: true, settings });
    } catch (error) {
      sendResponse({ success: false, error: error.message });
    }
  }

  async handleTogglePlugin(request, sendResponse) {
    try {
      // This is handled by the popup script, just acknowledge
      sendResponse({ success: true });
    } catch (error) {
      sendResponse({ success: false, error: error.message });
    }
  }

  async syncPluginStateWithTab(tabId) {
    try {
      // Wait a bit for the content script to load
      setTimeout(async () => {
        const settings = await this.getStoredSettings();
        const isEnabled = settings.pluginEnabled !== false;
        
        // Send plugin state to the tab
        chrome.tabs.sendMessage(tabId, {
          action: 'togglePlugin',
          enabled: isEnabled
        }).catch(() => {
          // Ignore errors for tabs without content script
        });
      }, 1000); // Wait 1 second for content script to initialize
    } catch (error) {
      console.error('Error syncing plugin state with tab:', error);
    }
  }

  async syncPluginStateWithAllTabs() {
    try {
      // Wait a bit for the extension to fully initialize
      setTimeout(async () => {
        const tabs = await chrome.tabs.query({});
        tabs.forEach(tab => {
          if (tab.url && tab.url.startsWith('http')) {
            this.syncPluginStateWithTab(tab.id);
          }
        });
      }, 2000); // Wait 2 seconds for extension to fully initialize
    } catch (error) {
      console.error('Error syncing plugin state with all tabs:', error);
    }
  }

  async getStoredRules() {
    try {
      const result = await chrome.storage.local.get('userRules');
      return result.userRules || { general: '' };
    } catch (error) {
      console.error('Error getting stored rules:', error);
      return { general: '' };
    }
  }

  async getStoredSettings() {
    try {
      const result = await chrome.storage.local.get('userSettings');
      const defaultSettings = {
        provider: 'openai',
        apiKey: '',
        maxSamples: 3,
        pluginEnabled: true
      };
      
      if (result.userSettings) {
        return { ...defaultSettings, ...result.userSettings };
      }
      
      return defaultSettings;
    } catch (error) {
      console.error('Error getting stored settings:', error);
      return {
        provider: 'openai',
        apiKey: '',
        maxSamples: 3,
        pluginEnabled: true
      };
    }
  }

  checkRateLimit(tabId) {
    const now = Date.now();
    const windowMs = 60000; // 1 minute window
    const maxRequests = 10; // Max 10 requests per minute per tab
    
    if (!this.rateLimiter.has(tabId)) {
      this.rateLimiter.set(tabId, []);
    }
    
    const requests = this.rateLimiter.get(tabId);
    
    // Remove old requests outside the window
    const validRequests = requests.filter(time => now - time < windowMs);
    
    if (validRequests.length >= maxRequests) {
      return false; // Rate limit exceeded
    }
    
    // Add current request
    validRequests.push(now);
    this.rateLimiter.set(tabId, validRequests);
    
    return true; // Request allowed
  }

  cleanupRateLimiter() {
    const now = Date.now();
    const windowMs = 60000; // 1 minute window
    
    for (const [tabId, requests] of this.rateLimiter.entries()) {
      const validRequests = requests.filter(time => now - time < windowMs);
      if (validRequests.length === 0) {
        this.rateLimiter.delete(tabId);
      } else {
        this.rateLimiter.set(tabId, validRequests);
      }
    }
  }
}

// AI Provider Classes
class OpenAIProvider {
  async generateRewrites(text, tone, length, apiKey, rules = {}, maxSamples = 3) {
    const prompt = this.buildPrompt(text, tone, length, rules, maxSamples);
    
    const response = await fetch('https://api.openai.com/v1/chat/completions', {
      method: 'POST',
      headers: {
        'Authorization': `Bearer ${apiKey}`,
        'Content-Type': 'application/json'
      },
      body: JSON.stringify({
        model: 'gpt-3.5-turbo',
        messages: [
          {
            role: 'user',
            content: prompt
          }
        ],
        max_tokens: 300,
        temperature: 0.7
      })
    });

    if (!response.ok) {
      const error = await response.json();
      throw new Error(error.error?.message || 'OpenAI API request failed');
    }

    const data = await response.json();
    return this.parseResponse(data.choices[0].message.content, maxSamples);
  }

  buildPrompt(text, tone, length, rules = {}, maxSamples = 3) {
    let prompt = `Please rewrite the following text in ${maxSamples} different ways with a ${tone} tone. `;
    
    // Add length instructions
    switch (length) {
      case 'shorter':
        prompt += 'Make the rewrites shorter than the original. ';
        break;
      case 'longer':
        prompt += 'Make the rewrites longer and more detailed than the original. ';
        break;
      default:
        prompt += 'Keep the rewrites approximately the same length as the original. ';
    }

    // Add general rules if they exist
    if (rules.general && rules.general.trim()) {
      prompt += `\n\nIMPORTANT: Use these guidelines to inform your rewrites, but NEVER include them in your response. Guidelines: ${rules.general}`;
    }

    const numberedList = Array.from({length: maxSamples}, (_, i) => `${i + 1}:`).join(', ');
    prompt += `\n\nOriginal text: "${text}"\n\nCRITICAL: Provide exactly ${maxSamples} alternative versions, each on a separate line starting with ${numberedList}. Your response must ONLY contain the ${maxSamples} rewritten versions - no explanations, no guidelines, no additional text.`;
    
    return prompt;
  }

  parseResponse(response, maxSamples = 3) {
    const lines = response.split('\n').filter(line => line.trim());
    const rewrites = [];
    
    for (const line of lines) {
      const match = line.match(/^\d+:\s*(.+)$/);
      if (match) {
        const rewriteText = match[1].trim();
        // Filter out any text that looks like guidelines or instructions
        if (!this.isGuidelineText(rewriteText)) {
          rewrites.push(rewriteText);
        }
      }
    }
    
    // Fallback if parsing fails
    if (rewrites.length === 0) {
      const sentences = response.split(/[.!?]+/).filter(s => s.trim().length > 10);
      return sentences.slice(0, maxSamples).filter(s => !this.isGuidelineText(s));
    }
    
    return rewrites.slice(0, maxSamples);
  }

  isGuidelineText(text) {
    const guidelinePatterns = [
      /guidelines?/i,
      /rules?/i,
      /instructions?/i,
      /ensure/i,
      /imperative/i,
      /crucial/i,
      /strict adherence/i,
      /follow/i,
      /adhere/i,
      /custom guidelines/i,
      /project goals/i,
      /desired outcomes/i
    ];
    
    return guidelinePatterns.some(pattern => pattern.test(text));
  }
}



// Initialize the background service worker
const background = new AIRewriterBackground();
