// AI Content Rewriter - Popup Script
class PopupManager {
  constructor() {
    this.elements = {
      // Tab elements
      tabs: document.querySelectorAll('.tab'),
      tabContents: document.querySelectorAll('.tab-content'),
      
      // Main tab elements
      providerSelect: document.getElementById('provider-select'),
      apiKeyInput: document.getElementById('api-key'),
      apiKeyToggle: document.getElementById('toggle-api-key'),
              maxSamplesSelect: document.getElementById('max-samples'),
        saveButton: document.getElementById('save-settings'),
      saveText: document.getElementById('save-text'),
      saveLoading: document.getElementById('save-loading'),
      statusIndicator: document.getElementById('status-indicator'),
      statusText: document.getElementById('status-text'),
      
      // Rules tab elements
      generalRules: document.getElementById('general-rules'),
      rulesPreview: document.getElementById('rules-preview'),
      saveRulesButton: document.getElementById('save-rules'),
      saveRulesText: document.getElementById('save-rules-text'),
      saveRulesLoading: document.getElementById('save-rules-loading'),
      exportRulesButton: document.getElementById('export-rules'),
      importRulesButton: document.getElementById('import-rules'),
      importFileInput: document.getElementById('import-file'),
      pluginEnabledCheckbox: document.getElementById('plugin-enabled')
    };
    
    // Set default provider immediately
    this.elements.providerSelect.value = 'openai';
    this.init();
  }

  async init() {
    this.setupEventListeners();
    this.setupTabSwitching();
    
    // Ensure provider is set to openai by default
    this.elements.providerSelect.value = 'openai';
    
    await this.loadSettings();
    await this.loadRules();
    this.updateRulesPreview();
  }

  setupTabSwitching() {
    this.elements.tabs.forEach(tab => {
      tab.addEventListener('click', () => {
        const targetTab = tab.getAttribute('data-tab');
        this.switchTab(targetTab);
      });
    });
  }

  switchTab(targetTab) {
    // Update tab buttons
    this.elements.tabs.forEach(tab => {
      tab.classList.remove('active');
      if (tab.getAttribute('data-tab') === targetTab) {
        tab.classList.add('active');
      }
    });

    // Update tab content
    this.elements.tabContents.forEach(content => {
      content.classList.remove('active');
      if (content.id === `${targetTab}-tab`) {
        content.classList.add('active');
      }
    });
  }

  setupEventListeners() {
    // API key visibility toggle
    this.elements.apiKeyToggle.addEventListener('click', () => {
      const input = this.elements.apiKeyInput;
      const toggle = this.elements.apiKeyToggle;
      
      if (input.type === 'password') {
        input.type = 'text';
        toggle.textContent = 'Hide';
      } else {
        input.type = 'password';
        toggle.textContent = 'Show';
      }
    });

    // Provider change handler
    this.elements.providerSelect.addEventListener('change', () => {
      this.updateApiKeyPlaceholder();
      this.validateSettings();
    });

    // API key input handler
    this.elements.apiKeyInput.addEventListener('input', () => {
      this.validateSettings();
    });

    // Save settings
    this.elements.saveButton.addEventListener('click', () => {
      this.saveSettings();
    });

    // Save rules
    this.elements.saveRulesButton.addEventListener('click', () => {
      this.saveRules();
    });

    // Export rules
    this.elements.exportRulesButton.addEventListener('click', () => {
      this.exportRules();
    });

    // Import rules
    this.elements.importRulesButton.addEventListener('click', () => {
      this.elements.importFileInput.click();
    });

    // Import file change
    this.elements.importFileInput.addEventListener('change', (e) => {
      this.importRules(e);
    });

    // Rules input handler for real-time preview
    this.elements.generalRules.addEventListener('input', () => {
      this.updateRulesPreview();
    });

    // Plugin enabled checkbox handler
    this.elements.pluginEnabledCheckbox.addEventListener('change', () => {
      this.savePluginState();
    });

    // Real-time validation
    [this.elements.apiKeyInput, this.elements.providerSelect].forEach(element => {
      element.addEventListener('change', () => this.validateSettings());
      element.addEventListener('input', () => this.validateSettings());
    });
  }

  updateApiKeyPlaceholder() {
    const provider = this.elements.providerSelect.value;
    const placeholders = {
      openai: 'sk-...'
    };
    
    this.elements.apiKeyInput.placeholder = `Enter your ${provider.toUpperCase()} API key (${placeholders[provider]})`;
  }

  async loadSettings() {
    try {
      const response = await chrome.runtime.sendMessage({ action: 'getSettings' });
      
      if (response && response.success) {
        const settings = response.settings;
        
        this.elements.providerSelect.value = settings.provider || 'openai';
        this.elements.apiKeyInput.value = settings.apiKey || '';
        this.elements.maxSamplesSelect.value = settings.maxSamples || 3;
        this.elements.pluginEnabledCheckbox.checked = settings.pluginEnabled !== false;
        
        this.updateApiKeyPlaceholder();
        this.validateSettings();
        
        // Sync plugin state with all open tabs
        this.syncPluginStateWithTabs(settings.pluginEnabled !== false);
      } else {
        console.error('Failed to get settings response:', response);
        // Set default values if no response
        this.elements.providerSelect.value = 'openai';
        this.elements.maxSamplesSelect.value = 3;
        this.elements.pluginEnabledCheckbox.checked = true;
        this.updateApiKeyPlaceholder();
        this.validateSettings();
        
        // Sync plugin state with all open tabs (default enabled)
        this.syncPluginStateWithTabs(true);
      }
    } catch (error) {
      this.showStatus('Failed to load settings', 'error');
      console.error('Load settings error:', error);
      // Set default values on error
      this.elements.providerSelect.value = 'openai';
      this.elements.maxSamplesSelect.value = 3;
      this.elements.pluginEnabledCheckbox.checked = true;
      this.updateApiKeyPlaceholder();
      this.validateSettings();
      
      // Sync plugin state with all open tabs (default enabled)
      this.syncPluginStateWithTabs(true);
    }
  }

  async loadRules() {
    try {
      const response = await chrome.runtime.sendMessage({ action: 'getRules' });
      
      if (response && response.success) {
        const rules = response.rules;
        
        // Load rules into form fields
        this.elements.generalRules.value = rules.general || '';
        
        this.updateRulesPreview();
      } else {
        console.error('Failed to get rules response:', response);
      }
    } catch (error) {
      console.error('Load rules error:', error);
    }
  }

  async saveRules() {
    const saveButton = this.elements.saveRulesButton;
    const saveText = this.elements.saveRulesText;
    const saveLoading = this.elements.saveRulesLoading;
    
    // Show loading state
    saveButton.disabled = true;
    saveText.classList.add('hidden');
    saveLoading.classList.remove('hidden');
    
    try {
      const rules = {
        general: this.elements.generalRules.value.trim()
      };

      const response = await chrome.runtime.sendMessage({
        action: 'saveRules',
        rules: rules
      });

      if (response && response.success) {
        this.showStatus('Rules saved successfully!', 'success');
        this.updateRulesPreview();
      } else {
        const errorMsg = response?.error || 'Failed to save rules';
        throw new Error(errorMsg);
      }
    } catch (error) {
      this.showStatus(error.message, 'error');
      console.error('Save rules error:', error);
    } finally {
      // Reset button state
      saveButton.disabled = false;
      saveText.classList.remove('hidden');
      saveLoading.classList.add('hidden');
    }
  }

  exportRules() {
    try {
      const rules = {
        general: this.elements.generalRules.value.trim()
      };

      // Create a blob with the rules data
      const dataStr = JSON.stringify(rules, null, 2);
      const dataBlob = new Blob([dataStr], { type: 'application/json' });

      // Create download link
      const link = document.createElement('a');
      link.href = URL.createObjectURL(dataBlob);
      link.download = `ai-rewriter-rules-${new Date().toISOString().split('T')[0]}.json`;
      
      // Trigger download
      document.body.appendChild(link);
      link.click();
      document.body.removeChild(link);
      
      // Clean up
      URL.revokeObjectURL(link.href);
      
      this.showStatus('Rules exported successfully!', 'success');
    } catch (error) {
      this.showStatus('Failed to export rules', 'error');
      console.error('Export rules error:', error);
    }
  }

  async importRules(event) {
    try {
      const file = event.target.files[0];
      if (!file) return;

      const text = await file.text();
      const rules = JSON.parse(text);

      if (rules.general) {
        this.elements.generalRules.value = rules.general;
        this.updateRulesPreview();
        this.showStatus('Rules imported successfully!', 'success');
      } else {
        throw new Error('Invalid rules file format');
      }
    } catch (error) {
      this.showStatus('Failed to import rules: Invalid file format', 'error');
      console.error('Import rules error:', error);
    } finally {
      // Reset file input
      event.target.value = '';
    }
  }

  updateRulesPreview() {
    const rules = {
      general: this.elements.generalRules.value.trim()
    };

    let previewText = '<strong>Current Rules:</strong> ';
    
    if (rules.general) {
      // Truncate long rules for preview
      const truncatedRules = rules.general.length > 200 
        ? rules.general.substring(0, 200) + '...'
        : rules.general;
      
      previewText += `<br><br>${truncatedRules.replace(/\n/g, '<br>')}`;
    } else {
      previewText = '<strong>Current Rules:</strong> No custom rules configured. Default AI behavior will be used.';
    }

    this.elements.rulesPreview.innerHTML = previewText;
  }

  async saveSettings() {
    const saveButton = this.elements.saveButton;
    const saveText = this.elements.saveText;
    const saveLoading = this.elements.saveLoading;
    
    // Show loading state
    saveButton.disabled = true;
    saveText.classList.add('hidden');
    saveLoading.classList.remove('hidden');
    
    try {
      const settings = {
        provider: this.elements.providerSelect.value,
        apiKey: this.elements.apiKeyInput.value.trim(),
        maxSamples: parseInt(this.elements.maxSamplesSelect.value),
        pluginEnabled: this.elements.pluginEnabledCheckbox.checked
      };

      // Validate API key format
      if (!this.validateApiKeyFormat(settings.provider, settings.apiKey)) {
        throw new Error('Invalid API key format for selected provider');
      }

      const response = await chrome.runtime.sendMessage({
        action: 'saveSettings',
        settings: settings
      });

      if (response && response.success) {
        this.showStatus('Settings saved successfully!', 'success');
        
        // Test the API key
        setTimeout(() => {
          this.testApiConnection(settings.provider, settings.apiKey);
        }, 1000);
      } else {
        const errorMsg = response?.error || 'Failed to save settings - no response from background script';
        console.error('Save settings response:', response);
        throw new Error(errorMsg);
      }
    } catch (error) {
      this.showStatus(error.message, 'error');
      console.error('Save settings error:', error);
    } finally {
      // Reset button state
      saveButton.disabled = false;
      saveText.classList.remove('hidden');
      saveLoading.classList.add('hidden');
    }
  }

  validateApiKeyFormat(provider, apiKey) {
    if (!apiKey) return false;
    
    // Both providers only check sk- prefix
    return apiKey.startsWith('sk-');
  }

  async testApiConnection(provider, apiKey) {
    try {
      // Test with a simple rewrite request
      const response = await chrome.runtime.sendMessage({
        action: 'rewriteText',
        text: 'Test connection',
        tone: 'professional',
        length: 'same'
      });

      if (response.success) {
        this.showStatus('API connection verified!', 'success');
      } else {
        this.showStatus(`API connection failed: ${response.error}`, 'warning');
      }
    } catch (error) {
      this.showStatus('API key saved, but connection test failed', 'warning');
    }
  }

  validateSettings() {
    const provider = this.elements.providerSelect.value;
    const apiKey = this.elements.apiKeyInput.value.trim();
    
    if (!apiKey) {
      this.showStatus('API key required', 'warning');
      return false;
    }
    
          if (!this.validateApiKeyFormat(provider, apiKey)) {
      let hint = '';
      if (provider === 'openai') {
        hint = ' (Should start with "sk-")';
      } else if (provider === 'claude') {
        hint = ' (Should start with "sk-ant-")';
      }
      this.showStatus(`Invalid API key format${hint}`, 'error');
      return false;
    }
    
    this.hideStatus();
    return true;
  }



  showStatus(message, type = 'info') {
    this.elements.statusIndicator.className = `status-indicator status-${type}`;
    this.elements.statusText.textContent = message;
    this.elements.statusIndicator.classList.remove('hidden');
    
    // Auto-hide success messages
    if (type === 'success') {
      setTimeout(() => {
        this.hideStatus();
      }, 3000);
    }
  }

  hideStatus() {
    this.elements.statusIndicator.classList.add('hidden');
  }

  // Utility method to get provider documentation URLs
  getProviderDocUrl(provider) {
    const urls = {
      openai: 'https://platform.openai.com/api-keys'
    };
    return urls[provider] || '#';
  }

  // Handle external links for API key setup
  setupProviderHelp() {
    const provider = this.elements.providerSelect.value;
    const helpUrl = this.getProviderDocUrl(provider);
    
    // This would open the provider's API key page
    chrome.tabs.create({ url: helpUrl });
  }

  async savePluginState() {
    try {
      const isEnabled = this.elements.pluginEnabledCheckbox.checked;
      
      // Get current settings and update plugin state
      const currentSettings = await chrome.storage.local.get('userSettings');
      const updatedSettings = {
        ...currentSettings.userSettings,
        pluginEnabled: isEnabled
      };
      
      // Save updated settings
      await chrome.storage.local.set({ userSettings: updatedSettings });
      
      // Sync with all open tabs
      this.syncPluginStateWithTabs(isEnabled);
      
      this.showStatus(`Plugin ${isEnabled ? 'enabled' : 'disabled'}`, 'success');
    } catch (error) {
      console.error('Save plugin state error:', error);
      this.showStatus('Failed to save plugin state', 'error');
    }
  }

  syncPluginStateWithTabs(isEnabled) {
    // Send message to ALL tabs to show/hide buttons
    chrome.tabs.query({}, (tabs) => {
      tabs.forEach(tab => {
        chrome.tabs.sendMessage(tab.id, {
          action: 'togglePlugin',
          enabled: isEnabled
        }).catch(() => {
          // Ignore errors for tabs without content script
        });
      });
    });
  }
}

// Keyboard shortcuts handler
document.addEventListener('keydown', (e) => {
  // Ctrl/Cmd + S to save settings
  if ((e.ctrlKey || e.metaKey) && e.key === 's') {
    e.preventDefault();
    document.getElementById('save-settings').click();
  }
  
  // Escape to close popup
  if (e.key === 'Escape') {
    window.close();
  }
});

// Initialize popup when DOM is ready
if (document.readyState === 'loading') {
  document.addEventListener('DOMContentLoaded', () => new PopupManager());
} else {
  new PopupManager();
}

// Handle popup visibility for real-time updates
document.addEventListener('visibilitychange', () => {
  if (!document.hidden) {
    // Reload settings when popup becomes visible to sync with localStorage
    const popup = window.popupManager;
    if (popup) {
      popup.loadSettings();
    }
  }
});